package pl.ctrl;

import java.util.List;
import java.util.Set;

import javax.annotation.Resource;
import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.SessionScoped;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.validator.ValidatorException;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.transaction.UserTransaction;

import pl.model.Book;
import pl.model.Publisher;
import pl.model.exception.UniquenessConstraintViolation;

@SessionScoped @ManagedBean( name="publisherCtrl")
public class PublisherController {
  @PersistenceContext( unitName="BidirAssocApp")
  private EntityManager em;
  @Resource private UserTransaction ut;

  /**
   * Get a reference to the entity manager
   * 
   * @return reference to the entitiy manager
   */
  public EntityManager getEntityManager() {
    return this.em;
  }

  /**
   * Read all publishers from the database.
   * 
   * @return an instance of all the publisher entries found in the database.
   */
  public List<Publisher> getPublishers() {
    return Publisher.retrieveAll( em);
  }

  /**
   * Update the reference object by setting its property values to match the one
   * existing in the database for the specific instance, identified by the
   * primary key value.
   * 
   * @param publisher
   *          the reference to the Publisher instance to be "loaded" from
   *          database.
   */
  public void refreshObject( Publisher publisher) {
    Publisher foundPublisher = Publisher.retrieve( em,
        publisher.getName());
    publisher.setName( foundPublisher.getName());
    publisher.setAddress( foundPublisher.getAddress());
    publisher.setPublishedBooks( foundPublisher.getPublishedBooks());
  }

  /**
   * UI specific check for the name uniqueness constraint. It uses the
   * <code>Publisher.checkNameAsId</code> method to verify the existence in the
   * database of a publisher entry for the given name value.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param value
   *          the value to be checked - in this case is the publisher name to
   *          look for in the database
   * @throws ValidatorException
   */
  public void checkNameAsId( FacesContext context, UIComponent component,
      Object value) throws ValidatorException {
    String name = (String) value;
    try {
      Publisher.checkNameAsId( em, name);
    } catch ( UniquenessConstraintViolation e) {
      throw new ValidatorException( new FacesMessage(
          FacesMessage.SEVERITY_ERROR, e.getMessage(), e.getMessage()));
    }
  }

  /**
   * Create and persist a new Publisher instance.
   * 
   * @param name
   *          the name of the publisher to create
   * @param address
   *          the address of the publisher to create
   * @param publishedBooks
   *          the books published by the publisher to create
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String add( String name, String address, Set<Book> publishedBooks) {
    try {
      Publisher.add( em, ut, name, address, publishedBooks);
      // Enforce clearing the form after creating the Publisher row.
      // Without this, the form will show the latest completed data
      FacesContext facesContext = FacesContext.getCurrentInstance();
      facesContext.getExternalContext().getRequestMap().remove( "publisher");
    } catch ( EntityExistsException e) {
      try {
        ut.rollback();
      } catch ( Exception e1) {
        e1.printStackTrace();
      }
      e.printStackTrace();
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "create";
  }

  /**
   * Update a Publisher instance.
   * 
   * @param name
   *          the name of the publisher to update (the publisher will be
   *          identified in the database by using this value)
   * @param address
   *          the new value for the address property
   * @param the
   *          new value for the publishedBooks property
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String update( String name, String address, Set<Book> publishedBooks) {
    try {
      Publisher.update( em, ut, name, address, publishedBooks);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "update";
  }

  /**
   * Delete a Publisher entry from database.
   * 
   * @param name
   *          the name of the publisher to delete - used to uniquely identify
   *          the publisher entry.
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String destroy( String name) {
    try {
      Publisher.destroy( em, ut, name);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "delete";
  }
}
