package pl.model;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ViewScoped;
import javax.faces.validator.ValidatorException;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.FetchType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.JoinTable;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.Query;
import javax.persistence.Table;
import javax.persistence.TypedQuery;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;
import javax.validation.constraints.NotNull;

import pl.model.exception.UniquenessConstraintViolation;

@Entity @Table( name="books")
@ViewScoped @ManagedBean( name="book")
public class Book {
  @Id @Column( length=10)
  @NotNull( message="An ISBN value is required!")
  private String isbn;
  @Column( nullable=false)
  @NotNull( message="A title is required!")
  private String title;
  @Column( nullable=false)
  @NotNull( message="A year is required!")
  private Integer year;
  @ManyToOne( fetch=FetchType.EAGER)
  @JoinColumn( name="PUBLISHER_NAME")
  private Publisher publisher;
  @ManyToMany( fetch=FetchType.EAGER)
  @JoinTable( name="books_authors", 
    joinColumns={ @JoinColumn( name="BOOK_ISBN") }, 
    inverseJoinColumns = { @JoinColumn( name="AUTHOR_PERSONID") })
  private Set<Author> authors;

  /**
   * Default constructor, required for entity classes
   */
  public Book() {}

  /**
   * Constructor
   */
  public Book( String isbn, String title, Integer year, Publisher publisher,
      Set<Author> authors) {
    this.setIsbn( isbn);
    this.setTitle( title);
    this.setYear( year);
    this.setPublisher( publisher);
    this.setAuthors( authors);
  }

  /**
   * Getters and setters
   */
  public String getTitle() {
    return title;
  }

  public void setTitle( String title) {
    this.title = title;
  }

  public String getIsbn() {
    return isbn;
  }

  public void setIsbn( String isbn) {
    this.isbn = isbn;
  }

  public Integer getYear() {
    return year;
  }

  public void setYear( Integer year) {
    this.year = year;
  }

  public Publisher getPublisher() {
    return publisher;
  }

  public void setPublisher( Publisher publisher) {
    if ( this.publisher != null) {
      // remove this book from the published books of the old publisher
      this.publisher.removePublishedBook( this);
    }
    if ( publisher != null) {
      // add this book to the published books of the new publisher
      publisher.addPublishedBook( this);
    }
    this.publisher = publisher;
  }

  public Set<Author> getAuthors() {
    return authors;
  }

  public void setAuthors( Set<Author> authors) {
    Set<Author> oldAUthors = this.authors != null ? new HashSet<Author>(
        this.authors) : null;
    // remove old authors
    if ( oldAUthors != null) {
      for ( Author a : oldAUthors) {
        this.removeAuthor( a);
      }
    }
    // add new authors
    if ( authors != null) {
      for ( Author a : authors) {
        this.addAuthor( a);
      }
    } else {
      if ( this.authors != null) {
        this.authors.clear();
      }
    }
  }

  public void addAuthor( Author author) {
    if ( this.authors == null) {
      this.authors = new HashSet<Author>();
    }
    if ( !this.authors.contains( author)) {
      this.authors.add( author);
      author.addAuthoredBook( this);
    }
  }

  public void removeAuthor( Author author) {
    if ( this.authors != null && author != null
        && this.authors.contains( author)) {
      this.authors.remove( author);
      author.removeAuthoredBook( this);
    }
  }

  /**
   * Return the serialized author names used in the JSF views.
   * 
   * @return serialized author names for this book.
   */
  public String getAuthorNames() {
    String result = "";
    int i = 0, n = 0;
    if ( this.authors != null) {
      n = this.authors.size();
      for ( Author author : this.authors) {
        result += author.getName();
        if ( i < n - 1) {
          result += ", ";
        }
        i++;
      }
    }
    return result;
  }

  /**
   * Create a human readable serialisation.
   * 
   */
  public String toString() {
    String result = "{ isbn: '" + this.isbn + "', title:'" + this.title
        + "', year: " + this.year + ", publisher: "
        + (this.publisher != null ? this.publisher.getName() : "")
        + ", authors: [";
    if ( this.authors != null) {
      int i = 0, n = this.authors.size();
      for ( Author a : this.authors) {
        result += "'" + a.getName() + "'";
        if ( i < n - 1) {
          result += ", ";
        }
        i++;
      }
    }
    result += "]}";
    return result;
  }
  
  @Override
  public boolean equals( Object obj) {
    if (obj instanceof Book) {
      Book book = (Book) obj;
      return ( this.isbn.equals( book.isbn));
    } else return false;
  }

  /**
   * Check for the isbn uniqueness constraint by verifying the existence in the
   * database of a book entry for the given isbn value.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param isbn
   *          the isbn of the book to check if exists or not
   * @throws ValidatorException
   * @throws UniquenessConstraintViolation
   */
  public static void checkIsbnAsId( EntityManager em, String isbn)
      throws UniquenessConstraintViolation {
    Book book = Book.retrieve( em, isbn);
    // book was found, uniqueness constraint validation failed
    if ( book != null) {
      throw new UniquenessConstraintViolation(
          "There is already a book record with this ISBN!");
    }
  }

  /**
   * Retrieve all book records from the books table.
   * 
   * @param em
   *          reference to the entity manager
   * @return all Book records
   */
  public static List<Book> retrieveAll( EntityManager em) {
    TypedQuery<Book> query = em.createQuery( "SELECT b FROM Book b", Book.class);
    List<Book> books = query.getResultList();
    System.out.println( "Book.retrieveAll: " + books.size()
        + " books were loaded from DB.");
    return books;
  }

  /**
   * Retrieve a book record from the books table.
   * 
   * @param em
   *          reference to the entity manager
   * @param isbn
   *          the book's ISBN
   * @return the book with the given ISBN
   */
  public static Book retrieve( EntityManager em, String isbn) {
    Book book = em.find( Book.class, isbn);
    if ( book != null) {
      System.out.println( "Book.retrieve: loaded book " + book);
    }
    return book;
  }

  /**
   * Create a Book instance.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the book to create
   * @param title
   *          the title value of the book to create
   * @param year
   *          the year value of the book to create
   * @param publisher
   *          the publisher value of the book to create
   * @param authors
   *          the authors for the book to create
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void add( EntityManager em, UserTransaction ut, String isbn,
      String title, Integer year, Publisher publisher, Set<Author> authors)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException, EntityExistsException {
    ut.begin();
    Book book = new Book( isbn, title, year, publisher, authors);
    em.persist( book);
    if ( publisher != null) {
      em.merge( publisher);
    }
    if ( authors != null) {
      for ( Author a : authors) {
        em.merge( a);
      }
    }
    ut.commit();
    System.out.println( "Book.add: the book " + book + " was saved.");
  }

  /**
   * Update a Book instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the book to update
   * @param title
   *          the title of the book to update
   * @param year
   *          the year of the book to update
   * @param publisher
   *          the publisher of the book to update
   * @param authors
   *          the authors for the book to update
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void update( EntityManager em, UserTransaction ut, String isbn,
      String title, Integer year, Publisher publisher, Set<Author> authors)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Book book = em.find( Book.class, isbn);
    if ( book == null) {
      throw new EntityNotFoundException( "The Book with ISBN = " + isbn
          + " was not found!");
    }
    Set<Author> oldAuthors = book.getAuthors();
    Publisher oldPublisher = book.getPublisher();
    if ( publisher != null && !publisher.equals( book.publisher)) {
      book.setPublisher( publisher);
    }
    if ( title != null && !title.equals( book.title)) {
      book.setTitle( title);
    }
    if ( year != null && !year.equals( book.year)) {
      book.setYear( year);
    }
    if ( authors != null && !authors.equals( book.authors)) {
      book.setAuthors( authors);
    }
    // merge managed entities in the memory, also update the cached versions
    // and the entity changes are reflected next time when used
    if ( oldPublisher != null) {
      em.merge( oldPublisher);
    }
    if ( publisher != null) {
      em.merge( publisher);
    }
    if ( oldAuthors != null) {
      for ( Author a : oldAuthors) {
        em.merge( a);
      }
    }
    if ( authors != null) {
      for ( Author a : authors) {
        em.merge( a);
      }
    }
    ut.commit();
    System.out.println( "Book.update: the book " + book + " was updated.");
  }

  /**
   * Delete a Book instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the book to delete
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void destroy( EntityManager em, UserTransaction ut, String isbn)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Book book = em.find( Book.class, isbn);
    book.setPublisher( null);
    book.setAuthors( null);
    em.remove( book);
    ut.commit();
    System.out.println( "Book.destroy: the book " + book + " was deleted.");
  }

  /**
   * Clear all entries from the <code>books</code> table
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void clearData( EntityManager em, UserTransaction ut)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Query deleteQuery = em.createQuery( "DELETE FROM Book");
    deleteQuery.executeUpdate();
    ut.commit();
  }
}