package pl.ctrl;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Resource;
import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.SessionScoped;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.validator.ValidatorException;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;

import pl.model.Book;
import pl.model.BookCategoryEL;
import pl.model.LanguageEL;
import pl.model.PublicationFormEL;
import pl.model.exception.UniquenessConstraintViolation;

@SessionScoped @ManagedBean( name="bookCtrl")
public class BookController {
  @PersistenceContext( unitName="EnumerationApp")
  private EntityManager em;
  @Resource private UserTransaction ut;

  /**
   * Read the list of all books from the database.
   * 
   * @return an instance of all the book entries found in the database.
   */
  public List<Book> getBooks() {
    return Book.retrieveAll( em);
  }

  /**
   * Update the reference object by setting its property values to match the one
   * existing in the database for the specific instance, identified by the
   * primary key value.
   * 
   * @param book
   *          the reference to the Book instance to be "loaded" from database.
   */
  public void refreshObject( Book book) {
    Book foundBook = Book.retrieve( em, book.getIsbn());
    book.setTitle( foundBook.getTitle());
    book.setYear( foundBook.getYear());
    book.setOriginalLanguage( foundBook.getOriginalLanguage());
    book.setOtherAvailableLanguages( foundBook.getOtherAvailableLanguages());
    book.setCategory( foundBook.getCategory());
    book.setPublicationForms( foundBook.getPublicationForms());
  }

  /**
   * UI specific check for the isbn uniqueness constraint. It uses the
   * <code>Book.checkIsbnAsId</code> method to verify the existence in the
   * database of a book entry for the given isbn value.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param value
   *          the value to be checked - in this case is the isbn to look for in
   *          the database
   * @throws ValidatorException
   */
  public void checkIsbnAsId( FacesContext context, UIComponent component,
      Object value) throws ValidatorException {
    String isbn = (String) value;
    try {
      Book.checkIsbnAsId( em, isbn);
    } catch ( UniquenessConstraintViolation e) {
      throw new ValidatorException( new FacesMessage(
          FacesMessage.SEVERITY_ERROR, e.getMessage(), e.getMessage()));
    }
  }

  /**
   * Create and persist a new Book instance.
   * 
   * @param isbn
   *          the isbn of the book to create
   * @param title
   *          the title of the book to create
   * @param year
   *          the year of the book to create
   * @param origilalLanguage
   *          the new value for the origilalLanguage property
   * @param otherAvailableLanguages
   *          the new value for the otherAvailableLanguages property
   * @param category
   *          the new value for the category property
   * @param publicationForms
   *          the publication forms of the book to create
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String add( String isbn, String title, Integer year,
      LanguageEL originalLanguage, Set<String> otherAvailableLanguages,
      BookCategoryEL category, Set<String> publicationForms) {
    try {
      Set<PublicationFormEL> pfSet = new HashSet<PublicationFormEL>();
      // add publicationForms values
      for ( String pfVal : publicationForms) {
        pfSet.add( PublicationFormEL.valueOf( pfVal));
      }
      Set<LanguageEL> oalSet = new HashSet<LanguageEL>();
      // add publicationForms values
      for ( String oalVal : otherAvailableLanguages) {
        oalSet.add( LanguageEL.valueOf( oalVal));
      }
      Book.add( em, ut, isbn, title, year, originalLanguage, oalSet, category, pfSet);
      // Enforce clearing the form after creating the Book row.
      // Without this, the form will show the latest completed data
      FacesContext facesContext = FacesContext.getCurrentInstance();
      facesContext.getExternalContext().getRequestMap().remove( "book");
    } catch ( EntityExistsException e) {
      try {
        ut.rollback();
      } catch ( Exception e1) {
        e1.printStackTrace();
      }
      e.printStackTrace();
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "create";
  }

  /**
   * Update a Book instance.
   * 
   * @param isbn
   *          the isbn of the book to update (the book will be identified in the
   *          database by using this value)
   * @param title
   *          the new value for the title property
   * @param year
   *          the new value for the year property
   * @param origilalLanguage
   *          the new value for the origilalLanguage property
   * @param otherAvailableLanguages
   *          the new value for the otherAvailableLanguages property
   * @param category
   *          the new value for the category property
   * @param publicationForms
   *          the new value for publicationForms property
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String update( String isbn, String title, Integer year,
      LanguageEL origilalLanguage, Set<String> otherAvailableLanguages,
      BookCategoryEL category, Set<String> publicationForms) {
    try {
      Set<PublicationFormEL> pfSet = new HashSet<PublicationFormEL>();
      // add publicationForms values
      for ( String pfVal : publicationForms) {
        pfSet.add( PublicationFormEL.valueOf( pfVal));
      }
      Set<LanguageEL> oalSet = new HashSet<LanguageEL>();
      // add publicationForms values
      for ( String oalVal : otherAvailableLanguages) {
        oalSet.add( LanguageEL.valueOf( oalVal));
      }
      Book.update( em, ut, isbn, title, year, origilalLanguage, oalSet, 
    		  category, pfSet);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "update";
  }

  /**
   * Delete a Book entry from database.
   * 
   * @param isbn
   *          the isbn of the book to delete - used to uniquely identify the
   *          book entry.
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   */
  public String destroy( String isbn) {
    try {
      Book.destroy( em, ut, isbn);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "delete";
  }

  /**
   * Clear/delete all entries from the <code>books</code> table
   * 
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   * 
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public String clearData() {
    try {
      Book.clearData( em, ut);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "index";
  }

  /**
   * Create test data (rows) in the <code>books</code> table
   * 
   * @return a string representing the view name to display after finishing the
   *         execution of this method.
   * 
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public String createTestData() {
    try {
      Book.createTestData( em, ut);
    } catch ( Exception e) {
      e.printStackTrace();
    }
    return "index";
  }
}
