package pl.model;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ViewScoped;
import javax.faces.model.SelectItem;
import javax.persistence.Column;
import javax.persistence.Convert;
import javax.persistence.Entity;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.Id;
import javax.persistence.Query;
import javax.persistence.Table;
import javax.persistence.TypedQuery;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;

import pl.model.exception.UniquenessConstraintViolation;

@Entity @Table( name="books")
@ViewScoped @ManagedBean( name="book")
public class Book {
  @Id @Column( length=10)
  @NotNull( message="An ISBN value is required!")
  private String isbn;
  @Column( nullable=false)
  @NotNull( message="A title is required!")
  private String title;
  @Column( nullable=false)
  @NotNull( message="A year is required!")
  private Integer year;
  @Column( nullable=false)
  @Enumerated( EnumType.STRING)
  @NotNull( message="An original language is required!")
  private LanguageEL originalLanguage;
  @Column( nullable=false)
  @Enumerated( EnumType.STRING)
  @NotNull( message="A category is required!")
  private BookCategoryEL category;
  @Convert( converter=pl.model.converter.OtherAvailableLanguagesConverter.class)
  private Set<LanguageEL> otherAvailableLanguages;
  @Column( nullable=false)
  @Convert( converter=pl.model.converter.PublicationFormsConverter.class)
  @Size( min=1, message="At least one publication form is required!")
  private Set<PublicationFormEL> publicationForms;

  /**
   * Default constructor, required for entity classes
   */
  public Book() {}

  /**
   * Constructor
   */
  public Book( String isbn, String title, Integer year,
      LanguageEL originalLanguage, Set<LanguageEL> otherAvailableLanguages,
      BookCategoryEL category, Set<PublicationFormEL> publicationForms) {
    this.setIsbn( isbn);
    this.setTitle( title);
    this.setYear( year);
    this.setOriginalLanguage( originalLanguage);
    this.setOtherAvailableLanguages( otherAvailableLanguages);
    this.setCategory( category);
    this.setPublicationForms( publicationForms);
  }

  public String getTitle() {
    return title;
  }

  public void setTitle( String title) {
    this.title = title;
  }

  public String getIsbn() {
    return isbn;
  }

  public void setIsbn( String isbn) {
    this.isbn = isbn;
  }

  public Integer getYear() {
    return year;
  }

  public void setYear( Integer year) {
    this.year = year;
  }

  public LanguageEL getOriginalLanguage() {
    return originalLanguage;
  }

  public void setOriginalLanguage( LanguageEL originalLanguage) {
    this.originalLanguage = originalLanguage;
  }

  public Set<LanguageEL> getOtherAvailableLanguages() {
    return this.otherAvailableLanguages;
  }

  public void setOtherAvailableLanguages(
      Set<LanguageEL> otherAvailableLanguages) {
    this.otherAvailableLanguages = otherAvailableLanguages;
  }

  public BookCategoryEL getCategory() {
    return this.category;
  }

  public void setCategory( BookCategoryEL category) {
    this.category = category;
  }

  public Set<PublicationFormEL> getPublicationForms() {
    return this.publicationForms;
  }

  public void setPublicationForms( Set<PublicationFormEL> publicationForms) {
    this.publicationForms = publicationForms;
  }

  /**
   * Get the otherAvailableLanguages property values as a string serialization.
   * 
   * @return a serialization of the otherAvailableLanguages, used to be
   *         displayed by the views
   */
  public String getOtherAvailableLanguagesValues() {
    String result = "";
    if ( this.otherAvailableLanguages != null) {
      int i = 0, n = this.otherAvailableLanguages.size();
      for ( LanguageEL oal : this.otherAvailableLanguages) {
        result += oal.getLabel();
        if ( i < n - 1) {
          result += ", ";
        }
        i++;
      }
    }
    return result;
  }

  /**
   * Create and return the set of items containing the languages which are
   * available for the book. The available languages are the literals of
   * LanguageEL enumeration.
   * 
   * @return the set of language items
   */
  public SelectItem[] getLanguageItems() {
    SelectItem[] items = new SelectItem[LanguageEL.values().length];
    int i = 0;
    for ( LanguageEL lang : LanguageEL.values()) {
      items[i++] = new SelectItem( lang.name(), lang.getLabel());
    }
    return items;
  }

  /**
   * Create and return the set of items containing the book categories which
   * are available for the book.
   * 
   * @return the set of book category items
   */
  public SelectItem[] getCategoryItems() {
    SelectItem[] items = new SelectItem[BookCategoryEL.values().length];
    int i = 0;
    for ( BookCategoryEL bk : BookCategoryEL.values()) {
      items[i++] = new SelectItem( bk.name(), bk.name().toLowerCase());
    }
    return items;
  }

  /**
   * Get the publicationForms property values as a string serialization.
   * 
   * @return a serialization of the publicationForms, used to be displayed by
   *         the views
   */
  public String getPublicationFormsValues() {
    String result = "";
    if ( this.publicationForms != null) {
      int i = 0, n = this.publicationForms.size();
      for ( PublicationFormEL pf : this.publicationForms) {
        result += pf.name().toLowerCase();
        if ( i < n - 1) {
          result += ", ";
        }
        i++;
      }
    }
    return result;
  }

  /**
   * Create and return the set of items containing the publication forms which
   * are available for the book.
   * 
   * @return the set of publication forms items
   */
  public SelectItem[] getPublicationFormsItems() {
    SelectItem[] items = new SelectItem[PublicationFormEL.values().length];
    int i = 0;
    for ( PublicationFormEL pf : PublicationFormEL.values()) {
      items[i++] = new SelectItem( pf.name(), pf.name().toLowerCase());
    }
    return items;
  }

  /**
   * Create a human readable serialization.
   */
  public String toString() {
    int i = 0, n = 0;
    String result = "{ isbn: '" + this.isbn + "', title:'" + this.title
        + "', year: " + this.year + ", originalLanguage: '"
        + this.originalLanguage.getLabel();
    result += "', otherAvailableLanguages: [";
    if ( this.otherAvailableLanguages != null){
      n = this.otherAvailableLanguages.size();
      for ( LanguageEL oal : this.otherAvailableLanguages) {
        result += "'" + oal.getLabel() + "'";
        if ( i < n - 1) {
          result += ", ";
        }
        i++;
      }
    }
    result += "]";
    result += ", category: '" + this.category.name().toLowerCase();
    result += "', publicationForms: [";
    i = 0;
    if ( this.publicationForms != null) {
      n = this.publicationForms.size();
      for ( PublicationFormEL pf : this.publicationForms) {
        result += "'" + pf.name().toLowerCase() + "'";
        if ( i < n - 1) {
          result += ", ";
        }
        i++;
      }
    }
    result += "]}";
    return result;
  }

  /**
   * Check for the isbn uniqueness constraint by verifying the existence in the
   * database of a book entry for the given isbn value.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param isbn
   *          the isbn of the book to check if exists or not
   * @throws UniquenessConstraintViolation
   */
  public static void checkIsbnAsId( EntityManager em, String isbn)
      throws UniquenessConstraintViolation {
    Book book = Book.retrieve( em, isbn);
    // book was found, uniqueness constraint validation failed
    if ( book != null) {
      throw new UniquenessConstraintViolation(
          "There is already a book record with this ISBN!");
    }
  }

  /**
   * Retrieve all book records from the books table.
   * 
   * @param em
   *          reference to the entity manager
   * @return the set of all Book records
   */
  public static List<Book> retrieveAll( EntityManager em) {
    TypedQuery<Book> query = em.createQuery( "SELECT b FROM Book b", Book.class);
    List<Book> books = query.getResultList();
    System.out.println( "Book.retrieveAll: " + books.size()
        + " books were loaded from DB.");
    return books;
  }

  /**
   * Read a Book entry from the <code>books</code> database table.
   * 
   * @param em
   *          reference to the entity manager
   * 
   * @return the book with the given ISBN entry found in the database.
   */
  public static Book retrieve( EntityManager em, String isbn) {
    Book book = em.find( Book.class, isbn);
    if ( book != null) {
      System.out.println( "Book.retrieve: loaded book " + book);
    }
    return book;
  }

  /**
   * Create a Book instance.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the book row to create
   * @param title
   *          the title value of the book row to create
   * @param year
   *          the year value of the book row to create
   * @param originalLanguage
   *          the originalLanguage value of the book row to create
   * @param otherAvailableLanguages
   *          the otherAvailableLanguages value of the book row to create
   * @param category
   *          the category value of the book row to create
   * @param publicationForms
   *          the publication forms of the book row to create
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void add( EntityManager em, UserTransaction ut, String isbn,
      String title, Integer year, LanguageEL originalLanguage,
      Set<LanguageEL> otherAvailableLanguages, BookCategoryEL category,
      Set<PublicationFormEL> publicationForms) throws NotSupportedException,
      SystemException, IllegalStateException, SecurityException,
      HeuristicMixedException, HeuristicRollbackException, RollbackException,
      EntityExistsException {
    ut.begin();
    Book book = new Book( isbn, title, year, originalLanguage,
        otherAvailableLanguages, category, publicationForms);
    em.persist( book);
    ut.commit();
    System.out.println( "Book.add: the book " + book + " was saved.");
  }

  /**
   * Update a Book instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN of the book to update
   * @param title
   *          the title of the book to update
   * @param year
   *          the year of the book to update
   * @param originalLanguage
   *          the originalLanguage of the book to update
   * @param otherAvailableLanguages
   *          the otherAvailableLanguages of the book to update
   * @param category
   *          the category of the book to update
   * @param publicationForms
   *          the publicationForms of the book to update
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void update( EntityManager em, UserTransaction ut, String isbn,
      String title, Integer year, LanguageEL originalLanguage,
      Set<LanguageEL> otherAvailableLanguages, BookCategoryEL category,
      Set<PublicationFormEL> publicationForms) throws NotSupportedException,
      SystemException, IllegalStateException, SecurityException,
      HeuristicMixedException, HeuristicRollbackException, RollbackException {
    ut.begin();
    Book book = em.find( Book.class, isbn);
    if ( book == null) {
      throw new EntityNotFoundException( "The Book with ISBN = " + isbn
          + " was not found!");
    }
    if ( title != null && !title.equals( book.title)) {
      book.setTitle( title);
    }
    if ( year != null && year != book.year) {
      book.setYear( year);
    }
    if ( originalLanguage != null
        && !originalLanguage.equals( book.originalLanguage)) {
      book.setOriginalLanguage( originalLanguage);
    }
    if ( otherAvailableLanguages != null
        && !otherAvailableLanguages.equals( book.otherAvailableLanguages)) {
      book.setOtherAvailableLanguages( otherAvailableLanguages);
    }
    if ( category != null && !category.equals( book.category)) {
      book.setCategory( category);
    }
    if ( publicationForms != null
        && !publicationForms.equals( book.publicationForms)) {
      book.setPublicationForms( publicationForms);
    }
    ut.commit();
    System.out.println( "Book.update: the book " + book + " was updated.");
  }

  /**
   * Delete a Book instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the book to delete
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void destroy( EntityManager em, UserTransaction ut, String isbn)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Book book = em.find( Book.class, isbn);
    em.remove( book);
    ut.commit();
    System.out.println( "Book.destroy: the book " + book + " was deleted.");
  }

  /**
   * Clear all entries from the <code>books</code> table
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void clearData( EntityManager em, UserTransaction ut)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Query deleteQuery = em.createQuery( "DELETE FROM Book");
    deleteQuery.executeUpdate();
    ut.commit();
  }

  /**
   * Create test data (rows) in the <code>books</code> table
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  @SuppressWarnings( "serial")
  public static void createTestData( EntityManager em, UserTransaction ut)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    Book book = null;
    // clear existing books, so no primary key duplicate conflicts appear
    Book.clearData( em, ut);
    ut.begin();
    book = new Book( "006251587X", "Weaving the Web", 2000, LanguageEL.EN,
        new HashSet<LanguageEL>() {
          {
            add( LanguageEL.DE);
            add( LanguageEL.FR);
          }
        }, BookCategoryEL.TEXTBOOK, new HashSet<PublicationFormEL>() {
          {
            add( PublicationFormEL.HARDCOVER);
            add( PublicationFormEL.PDF);
          }
        });
    em.persist( book);
    book = new Book( "0465026567", "Gdel, Escher, Bach", 1999, LanguageEL.FR,
        null, BookCategoryEL.OTHER, new HashSet<PublicationFormEL>() {
          {
            add( PublicationFormEL.PAPERBACK);
            add( PublicationFormEL.EPUB);
          }
        });
    em.persist( book);
    book = new Book( "0465030793", "I Am A Strange Loop", 2008, LanguageEL.ES,
        new HashSet<LanguageEL>() {
          {
            add( LanguageEL.EN);
            add( LanguageEL.DE);
          }
        }, BookCategoryEL.TEXTBOOK, new HashSet<PublicationFormEL>() {
          {
            add( PublicationFormEL.PDF);
            add( PublicationFormEL.EPUB);
          }
        });
    em.persist( book);
    ut.commit();
  }
}