package pl.model;

import java.util.List;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ViewScoped;
import javax.faces.model.SelectItem;
import javax.faces.validator.ValidatorException;
import javax.persistence.Column;
import javax.persistence.DiscriminatorColumn;
import javax.persistence.DiscriminatorType;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.Id;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.Query;
import javax.persistence.Table;
import javax.persistence.TypedQuery;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;
import javax.validation.constraints.NotNull;

import pl.model.exception.UniquenessConstraintViolation;

@Entity
@Table( name = "books")
@Inheritance( strategy = InheritanceType.SINGLE_TABLE)
@DiscriminatorColumn( name = "CATEGORY", discriminatorType = DiscriminatorType.STRING, length = 16)
@DiscriminatorValue( value = "BOOK")
@ViewScoped
@ManagedBean( name = "book")
public class Book {
  @Id
  @Column( length = 10)
  @NotNull( message = "An ISBN is required!")
  private String isbn;
  @Column( nullable = false)
  @NotNull( message = "A title is required!")
  private String title;
  @Column( nullable = false)
  @NotNull( message = "A year is required!")
  private Integer year;

  /**
   * Default constructor, required for entity classes
   */
  public Book() {
  }

  /**
   * Constructor
   */
  public Book( String isbn, String title, Integer year) {
    this.setIsbn( isbn);
    this.setTitle( title);
    this.setYear( year);
  }

  /**
   * Getters and setters
   */
  public String getTitle() {
    return title;
  }

  public void setTitle( String title) {
    this.title = title;
  }

  public String getIsbn() {
    return isbn;
  }

  public void setIsbn( String isbn) {
    this.isbn = isbn;
  }

  public Integer getYear() {
    return year;
  }

  public void setYear( Integer year) {
    this.year = year;
  }

  /**
   * Create a human readable serialisation.
   */
  public String toString() {
    String result = "{ isbn: '" + this.isbn + "', title:'" + this.title
        + "', year: " + this.year + "}";
    return result;
  }

  /**
   * Create and return the list of items containing the book types which are
   * available for the book.
   * 
   * @return the list of book type items
   */
  public SelectItem[] getTypeItems() {
    SelectItem[] items = new SelectItem[BookTypeEL.values().length];
    int i = 0;
    for ( BookTypeEL bt : BookTypeEL.values()) {
      items[i++] = new SelectItem( bt.name(), bt.getLabel());
    }
    return items;
  }

  @Override
  public boolean equals( Object obj) {
    if ( obj instanceof Book) {
      Book book = (Book) obj;
      return (this.isbn.equals( book.isbn));
    } else
      return false;
  }

  /**
   * Check for the isbn uniqueness constraint by verifying the existence in the
   * database of a book entry for the given isbn value.
   * 
   * @param context
   *          the faces context - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param component
   *          the UI component reference - used by the system when the method is
   *          automatically called from JSF facelets.
   * @param isbn
   *          the isbn of the book to check if exists or not
   * @throws ValidatorException
   * @throws UniquenessConstraintViolation
   */
  public static void checkIsbnAsId( EntityManager em, String isbn)
      throws UniquenessConstraintViolation {
    Book book = Book.retrieve( em, isbn);
    // book was found, uniqueness constraint validation failed
    if ( book != null) {
      throw new UniquenessConstraintViolation(
          "There is already a book record with this ISBN!");
    }
  }

  /**
   * Retrieve all book records from the books table.
   * 
   * @param em
   *          reference to the entity manager
   * @return the list of all Book records
   */
  public static List<Book> retrieveAll( EntityManager em) {
    TypedQuery<Book> query = em
        .createQuery( "SELECT b FROM Book b", Book.class);
    List<Book> books = query.getResultList();
    System.out.println( "Book.retrieveAll: " + books.size()
        + " books were loaded from DB.");
    return books;
  }

  /**
   * Retrieve a book record from the books table.
   * 
   * @param em
   *          reference to the entity manager
   * @param isbn
   *          the book's ISBN
   * @return the book with the given ISBN
   */
  public static Book retrieve( EntityManager em, String isbn) {
    Book book = em.find( Book.class, isbn);
    if ( book != null) {
      System.out.println( "Book.retrieve: loaded book " + book);
    }
    return book;
  }

  /**
   * Create a Book instance.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the book to create
   * @param title
   *          the title value of the book to create
   * @param year
   *          the year value of the book to create
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void add( EntityManager em, UserTransaction ut, String isbn,
      String title, Integer year) throws NotSupportedException,
      SystemException, IllegalStateException, SecurityException,
      HeuristicMixedException, HeuristicRollbackException, RollbackException,
      EntityExistsException {
    ut.begin();
    Book book = new Book( isbn, title, year);
    em.persist( book);
    ut.commit();
    System.out.println( "Book.add: the book " + book + " was saved.");
  }

  /**
   * Update a Book instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the book to update
   * @param title
   *          the title of the book to update
   * @param year
   *          the year of the book to update
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void update( EntityManager em, UserTransaction ut, String isbn,
      String title, Integer year) throws NotSupportedException,
      SystemException, IllegalStateException, SecurityException,
      HeuristicMixedException, HeuristicRollbackException, RollbackException {
    ut.begin();
    Book book = em.find( Book.class, isbn);
    if ( book == null) {
      throw new EntityNotFoundException( "The Book with ISBN = " + isbn
          + " was not found!");
    }
    if ( title != null && !title.equals( book.title)) {
      book.setTitle( title);
    }
    if ( year != null && !year.equals( book.year)) {
      book.setYear( year);
    }
    ut.commit();
    System.out.println( "Book.update: the book " + book + " was updated.");
  }

  /**
   * Delete a Book instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the book to delete
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void destroy( EntityManager em, UserTransaction ut, String isbn)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Book book = em.find( Book.class, isbn);
    em.remove( book);
    ut.commit();
    System.out.println( "Book.destroy: the book " + book + " was deleted.");
  }

  /**
   * Clear all entries from the <code>books</code> table
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void clearData( EntityManager em, UserTransaction ut)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    Query deleteQuery = em.createQuery( "DELETE FROM Book");
    deleteQuery.executeUpdate();
    ut.commit();
  }
}