package pl.model;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ViewScoped;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.UserTransaction;
import javax.validation.constraints.NotNull;

@Entity
@DiscriminatorValue( value = "TEXTBOOK")
@ViewScoped
@ManagedBean( name = "textBook")
public class TextBook extends Book {
  @NotNull( message = "A subject area value is required")
  String subjectArea;

  /**
   * Default constructor, required for entity classes
   */
  public TextBook() {
  }

  /**
   * Constructor
   */
  public TextBook( String isbn, String title, Integer year, String subjectArea) {
    super( isbn, title, year);
    this.setSubjectArea( subjectArea);
  }

  /**
   * Getters and setters
   */
  public String getSubjectArea() {
    return subjectArea;
  }

  public void setSubjectArea( String subjectArea) {
    this.subjectArea = subjectArea;
  }

  /**
   * Retrieve a TextBook record from the database.
   * 
   * @param em
   *          reference to the entity manager
   * @param isbn
   *          the book's ISBN
   * @return the text book with the given ISBN
   */
  public static TextBook retrieve( EntityManager em, String isbn) {
    TextBook textBook = em.find( TextBook.class, isbn);
    if ( textBook != null) {
      System.out.println( "TextBook.retrieve: loaded text book " + textBook);
    }
    return textBook;
  }

  /**
   * Create a human readable serialisation.
   */
  public String toString() {
    String result = super.toString();
    result = result.substring( 0, result.length() - 2);
    return result + ", subjectArea: '" + this.subjectArea + "'}";
  }

  /**
   * Create a TextBook instance.
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the text book to create
   * @param title
   *          the title value of the text book to create
   * @param year
   *          the year value of the text book to create
   * @param subjectArea
   *          the subjectArea of the text book to create
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void add( EntityManager em, UserTransaction ut, String isbn,
      String title, Integer year, String subjectArea)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException, EntityExistsException {
    ut.begin();
    TextBook textBook = new TextBook( isbn, title, year, subjectArea);
    em.persist( textBook);
    ut.commit();
    System.out.println( "TextBook.add: the text book " + textBook
        + " was saved.");
  }

  /**
   * Update a TextBook instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the text book to update
   * @param title
   *          the title value of the text book to update
   * @param year
   *          the year value of the text book to update
   * @param subjectArea
   *          the subjectArea value of the text book to update
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void update( EntityManager em, UserTransaction ut, String isbn,
      String title, Integer year, String subjectArea)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    TextBook book = em.find( TextBook.class, isbn);
    if ( book == null) {
      throw new EntityNotFoundException( "The Book with ISBN = " + isbn
          + " was not found!");
    }
    if ( title != null && !title.equals( book.getTitle())) {
      book.setTitle( title);
    }
    if ( year != null && !year.equals( book.getYear())) {
      book.setYear( year);
    }
    if ( subjectArea != null && !subjectArea.equals( book.getSubjectArea())) {
      book.setSubjectArea( subjectArea);
    }
    ut.commit();
    System.out.println( "TextBook.update: the text book " + book
        + " was updated.");
  }

  /**
   * Delete a TextBook instance
   * 
   * @param em
   *          reference to the entity manager
   * @param ut
   *          reference to the user transaction
   * @param isbn
   *          the ISBN value of the text book to delete
   * @throws NotSupportedException
   * @throws SystemException
   * @throws IllegalStateException
   * @throws SecurityException
   * @throws HeuristicMixedException
   * @throws HeuristicRollbackException
   * @throws RollbackException
   */
  public static void destroy( EntityManager em, UserTransaction ut, String isbn)
      throws NotSupportedException, SystemException, IllegalStateException,
      SecurityException, HeuristicMixedException, HeuristicRollbackException,
      RollbackException {
    ut.begin();
    TextBook textBook = em.find( TextBook.class, isbn);
    em.remove( textBook);
    ut.commit();
    System.out.println( "TextBook.destroy: the text book " + textBook
        + " was deleted.");
  }
}
